<?php
// verify_pin.php
declare(strict_types=1);
session_start();
ini_set('display_errors','1'); error_reporting(E_ALL);

require_once __DIR__ . '/includes/db.php'; // adjust if path differs

header('Content-Type: application/json; charset=utf-8');

// simple helpers
function fail($msg) { echo json_encode(['success'=>false,'error'=>$msg]); exit; }
function ok()      { echo json_encode(['success'=>true]); exit; }

$uid = (int)($_SESSION['uid'] ?? 0);
if ($uid <= 0) fail('Not authenticated');

// CSRF check
$csrf_post = $_POST['csrf'] ?? '';
if (empty($csrf_post) || !hash_equals($_SESSION['csrf'] ?? '', $csrf_post)) {
    fail('Invalid CSRF');
}

$pin = trim((string)($_POST['pin'] ?? ''));
if (!preg_match('/^\d{4,6}$/', $pin)) fail('PIN must be 4–6 digits');

try {
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    $stmt = $pdo->prepare("SELECT transaction_pin_hash, pin_attempts, is_locked FROM users WHERE id = ? FOR UPDATE");
    $stmt->execute([$uid]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) fail('User not found');

    if ((int)$user['is_locked'] === 1) {
        fail('Account locked due to too many wrong PIN attempts');
    }

    if (!password_verify($pin, $user['transaction_pin_hash'])) {
        $newAttempts = ((int)$user['pin_attempts']) + 1;
        $lock = ($newAttempts >= 3) ? 1 : 0;
        $stmt = $pdo->prepare("UPDATE users SET pin_attempts = ?, is_locked = ? WHERE id = ?");
        $stmt->execute([$newAttempts, $lock, $uid]);

        if ($lock) {
            fail('Too many wrong PIN attempts — your account has been locked.');
        }

        fail('Invalid transaction PIN');
    }

    // correct PIN
    $stmt = $pdo->prepare("UPDATE users SET pin_attempts = 0 WHERE id = ?");
    $stmt->execute([$uid]);

    // store short-lived verification mark
    $_SESSION['pin_verified_at'] = time();

    ok();

} catch (Throwable $e) {
    error_log("verify_pin error: " . $e->getMessage());
    fail('Internal error');
}
